{Freeware for noncommercial use}

{
Version 1.31

Copyright by Steffen Kirbach, Germany
 kirbach@sksoft.de
 http://www.sksoft.de
}

unit BFH;

interface

uses
  Windows, Classes, SysUtils;

type
  EWeakKey = class(Exception);
  TOnRenameFile = procedure(Sender: TObject; const Path: String;
                            var Name: String) of object;
  TBFH = class(TComponent)
  private
    fulCBCLeft: LongInt;
    fulCBCRight: LongInt;
    fFastFileCrypt: Boolean;
    fSecureFileName: Boolean;
    fOnRenameFile: TOnRenameFile;
    procedure SetKeyword(Value: String);
  protected
    { Protected-Deklarationen }
  public
    OriginalName: ShortString;
    OriginalSize: LongInt;
    OriginalDate: LongInt;
    function GetRandomFileName: String;
    // Public Stream
    function CheckBFHStream(CryptStream: TStream): Boolean;
    procedure EncryptStream(PlainStream, CryptStream: TStream);
    function DecryptStream(CryptStream, PlainStream: TStream): Boolean;
    // Public File
    function CheckBFHFile(CryptFileName: String): Boolean;
    procedure EncryptFile(PlainFileName: String);
    function DecryptFile(CryptFileName: String): Boolean;
    function WriteOrigInfo(CryptFileName: String): Boolean;
  published
    property Keyword: String write SetKeyword stored False;
    property FastFileCrypt: Boolean read fFastFileCrypt write fFastFileCrypt;
    property SecureFileName: Boolean read fSecureFileName write fSecureFileName;
    property OnRenameFile: TOnRenameFile read fOnRenameFile write fOnRenameFile;
  end;

procedure Register;

implementation

uses
  CRC, Blowfish;

type
  TBFHHeader = packed record
                        Signature: Array[0..2] of Char;
                        Version: Byte;
                        CRC32: LongInt;
                        ulCBCLeft, ulCBCRight: LongInt;
                      end;
  TOrigInfo  = packed record
                        Name: Array[0..255] of Char;
                        Size: LongInt;
                        Date: LongInt;
                      end;
  TCRCBuffer = Array[0..255] of Byte;

const
  BFHSignature   = 'BFH';
  BFHModulo      = 8;
  BufferSize     = $2000 * BFHModulo;
  TempExt        = '.$$$';

const
  ErrorMsg: Array[1..6, 1..2] of String =
    ((Chr(LANG_ENGLISH),
      'Password was evaluated and rejected as weak.'),
     (Chr(LANG_GERMAN),
      'Kennwort wurde als schwach bewertet und abgelehnt.'),
     (Chr(LANG_FRENCH),
      'Le mot de passe a t valu et a rejet comme faible.'),
     (Chr(LANG_SPANISH),
      'La palabra de paso fue evaluada y rechaz como dbil.'),
     (Chr(LANG_ITALIAN),
      'La parola d''accesso  stata valutata e rifiutato come debole.'),
     (Chr(LANG_PORTUGUESE),
      'A senha foi avaliada e rejeitou como fraca.'));

{*** Intern *******************************************************************}

function Random32: LongInt;
begin
  Result:=(LongInt(Random($10000)) shl 16) or Random($10000);
end;

function GetErrorStr: String;
var
  Lang: Char;
  MsgID: Integer;
  i: Integer;
begin
  Lang:= Chr(GetUserDefaultLangID and $FF);
  MsgID:= Low(ErrorMsg);
  for i:= Low(ErrorMsg) to High(ErrorMsg) do
    if ErrorMsg[i, 1] = Lang then MsgID:= i;
  Result:= ErrorMsg[MsgID, 2];
end;

{*** Private ******************************************************************}

procedure TBFH.SetKeyword(Value: String);
var
  i: Integer;

  procedure KeyError;
  begin
    Blowfish_Done;
    raise EWeakKey.Create(GetErrorStr);
  end;

begin
  if Length(Value) = 0 then Blowfish_Done
  else
  begin
    for i:= 1 to Length(Value) div 2 do
      if Copy(Value, 1, i) = Copy(Value, i + 1, i) then KeyError;
    if not Blowfish_Init(PChar(Value), Length(Value), nil, 16) then KeyError;
  end;
end;

{*** Public *******************************************************************}

function TBFH.GetRandomFileName: String;
begin
  Result:= IntToHex(Random32, 8) + '.' + BFHSignature;
end;

{*** Public Stream ************************************************************}

function TBFH.CheckBFHStream(CryptStream: TStream): Boolean;
type
  TByteArray = Array[1..MaxInt] of Byte;
var
  BFHHeader: TBFHHeader;
  OrigInfo: TOrigInfo;
  CRCBuffer: TCRCBuffer absolute OrigInfo;
  CRCBufferFill: Integer;
  ulCBCLeft, ulCBCRight: LongInt;

{*** Old Beta Header *****************}
  procedure CheckOldBFHHeader;
  type
    TBFHHeaderO = packed record
                            Signature: Array[0..2] of Char;
                            Version: Byte;
                            OrigInfo: TOrigInfo;
                            OrigInfoCRC: LongInt;
                            ulCBCLeft, ulCBCRight: LongInt;
                          end;
  var
    BFHHeaderO: TBFHHeaderO;
    i: Integer;
    CRC32: LongInt;
  begin
    CryptStream.Position:= 0;
    CryptStream.ReadBuffer(BFHHeaderO, SizeOf(BFHHeaderO));
    ulCBCLeft := BFHHeaderO.ulCBCLeft;
    ulCBCRight:= BFHHeaderO.ulCBCRight;
    Blowfish_CBCDecrypt(@BFHHeaderO.OrigInfo,
                        SizeOf(BFHHeaderO.OrigInfo),
                        ulCBCLeft, ulCBCRight);
    CRC32:= 0;
    for i:= 1 to SizeOf(BFHHeaderO.OrigInfo) do
      CRC32:= UpdCRC_32(TByteArray(Pointer(@BFHHeaderO.OrigInfo)^)[i], CRC32);
    if BFHHeaderO.OrigInfoCRC = CRC32 then
    begin
      fulCBCLeft:= BFHHeaderO.ulCBCLeft;
      fulCBCRight:= BFHHeaderO.ulCBCRight;
      OriginalName:= String(BFHHeaderO.OrigInfo.Name);
      OriginalSize:= BFHHeaderO.OrigInfo.Size;
      OriginalDate:= BFHHeaderO.OrigInfo.Date;
      Result:= True;
    end;
  end;
{*************************************}

begin
  Result:= False;
  fulCBCLeft:= 0;
  fulCBCRight:= 0;
  OriginalName:= '';
  OriginalSize:= 0;
  OriginalDate:= 0;
  try
    CryptStream.Position:= 0;
    CryptStream.ReadBuffer(BFHHeader, SizeOf(TBFHHeader));
    if (BFHHeader.Signature = BFHSignature) then
    case BFHHeader.Version of
      0: CheckOldBFHHeader;
      1: begin
           CryptStream.ReadBuffer(OrigInfo, SizeOf(TOrigInfo));
           ulCBCLeft := BFHHeader.ulCBCLeft;
           ulCBCRight:= BFHHeader.ulCBCRight;
           Blowfish_CBCDecrypt(@OrigInfo, SizeOf(TOrigInfo),
                               ulCBCLeft, ulCBCRight);
           if BFHHeader.CRC32 = GetCRC_32(@CRCBuffer, SizeOf(TCRCBuffer)) then
           begin
             fulCBCLeft:= BFHHeader.ulCBCLeft;
             fulCBCRight:= BFHHeader.ulCBCRight;
             OriginalName:= String(OrigInfo.Name);
             OriginalSize:= OrigInfo.Size;
             OriginalDate:= OrigInfo.Date;
             Result:= True;
           end;
         end;
      2: begin
           CRCBufferFill:= CryptStream.Read(CRCBuffer, SizeOf(TCRCBuffer));
           ulCBCLeft := BFHHeader.ulCBCLeft;
           ulCBCRight:= BFHHeader.ulCBCRight;
           Blowfish_CBCDecrypt(@CRCBuffer, CRCBufferFill,
                               ulCBCLeft, ulCBCRight);
           if CryptStream.Position = CryptStream.Size then
             Dec(CRCBufferFill, CRCBuffer[Pred(CRCBufferFill)]);
           if BFHHeader.CRC32 = GetCRC_32(@CRCBuffer, CRCBufferFill) then
           begin
             fulCBCLeft:= BFHHeader.ulCBCLeft;
             fulCBCRight:= BFHHeader.ulCBCRight;
             Result:= True;
           end;
           CryptStream.Position:= SizeOf(TBFHHeader);
         end;
    end;
  except
    Result:= False;
  end;
end;

procedure TBFH.EncryptStream(PlainStream, CryptStream: TStream);
type
  TByteArray = Array[1..BufferSize] of Byte;
  TCopyBuffer = Array[Boolean] of Record
                                    ByteArray: ^TByteArray;
                                    BufferFill: LongInt;
                                  end;
var
  BFHHeader: TBFHHeader;
  OrigInfo: TOrigInfo;
  CRCBuffer: TCRCBuffer absolute OrigInfo;
  CRCBufferFill: Integer;
  CopyBuffer: TCopyBuffer;
  AltFlag: Boolean;
  ulCBCLeft, ulCBCRight: LongInt;
  FillByte: Byte;
  PlainEnd: Boolean;
  i: Integer;
begin
  PlainStream.Position:= 0;
  CryptStream.Position:= 0;

  BFHHeader.Signature:= BFHSignature;
  BFHHeader.ulCBCLeft:= Random32;
  BFHHeader.ulCBCRight:= Random32;

  ulCBCLeft:= BFHHeader.ulCBCLeft;
  ulCBCRight:= BFHHeader.ulCBCRight;

  New(CopyBuffer[False].ByteArray);
  New(CopyBuffer[True ].ByteArray);
  try
    AltFlag:= False;
    if Length(OriginalName) > 0 then
    begin
      BFHHeader.Version:= 1;
      with OrigInfo do
      begin
        for i:= Low(Name) to High(Name) do Name[i]:= Chr(Random($100));
        StrPCopy(Name, OriginalName);
        Size:= OriginalSize;
        Date:= OriginalDate;
      end;
      BFHHeader.CRC32:= GetCRC_32(@CRCBuffer, SizeOf(TCRCBuffer));
      Blowfish_CBCEncrypt(@OrigInfo, SizeOf(TOrigInfo), ulCBCLeft, ulCBCRight);
      ulCBCLeft:= BFHHeader.ulCBCLeft;
      ulCBCRight:= BFHHeader.ulCBCRight;
      with CopyBuffer[AltFlag] do
      begin
        Move(BFHHeader, ByteArray[1], SizeOf(TBFHHeader));
        BufferFill:= SizeOf(TBFHHeader);
        Move(OrigInfo, ByteArray[BufferFill + 1], SizeOf(TOrigInfo));
        Inc(BufferFill, SizeOf(TOrigInfo));
      end;
    end
    else
    begin
      BFHHeader.Version:= 2;
      CRCBufferFill:= PlainStream.Read(CRCBuffer, SizeOf(TCRCBuffer));
      BFHHeader.CRC32:= GetCRC_32(@CRCBuffer, CRCBufferFill);
      PlainStream.Position:= 0;
      with CopyBuffer[AltFlag] do
      begin
        Move(BFHHeader, ByteArray[1], SizeOf(TBFHHeader));
        BufferFill:= SizeOf(TBFHHeader);
      end;
    end;

    repeat
      AltFlag:= not AltFlag;
      with CopyBuffer[AltFlag] do
      begin
        with PlainStream do
        begin
          if (Size - Position) > (BufferSize - BFHModulo) then
            BufferFill:= Read(ByteArray^, BufferSize - BFHModulo)
          else BufferFill:= Read(ByteArray^, Size - Position);
          if Position = Size then
          begin
            for FillByte:= 1 to (BFHModulo - (BufferFill mod BFHModulo)) do
            begin
             Inc(BufferFill);
             ByteArray^[BufferFill]:= FillByte;
            end;
            PlainEnd:= True;
          end
          else PlainEnd:= False;
        end;
        Blowfish_CBCEncrypt(ByteArray, BufferFill, ulCBCLeft, ulCBCRight);
      end;
      with CopyBuffer[not AltFlag] do
        CryptStream.WriteBuffer(ByteArray^, BufferFill);
    until PlainEnd;
    with CopyBuffer[AltFlag] do CryptStream.WriteBuffer(ByteArray^, BufferFill);
  finally
    Dispose(CopyBuffer[False].ByteArray);
    Dispose(CopyBuffer[True ].ByteArray);
  end;
end;

function TBFH.DecryptStream(CryptStream, PlainStream: TStream): Boolean;
type
  TByteArray = Array[1..BufferSize] of Byte;
var
  ByteArray: ^TByteArray;
  BufferFill: LongInt;
  ulCBCLeft, ulCBCRight: LongInt;
begin
  if CheckBFHStream(CryptStream) then
  begin
    New(ByteArray);
    try
      ulCBCLeft := fulCBCLeft;
      ulCBCRight:= fulCBCRight;
      with CryptStream do
        while Position < Size do
        begin
          if (Size - Position) > BufferSize then
            BufferFill:= Read(ByteArray^, BufferSize)
          else BufferFill:= Read(ByteArray^, Size - Position);
          Blowfish_CBCDecrypt(ByteArray, BufferFill, ulCBCLeft, ulCBCRight);
          if Position = Size then Dec(BufferFill, ByteArray^[BufferFill]);
          PlainStream.WriteBuffer(ByteArray^, BufferFill);
        end;
    finally
      Dispose(ByteArray);
    end;
    PlainStream.Size:= PlainStream.Position;
    Result:= True;
  end
  else Result:= False;
end;

{*** Public File **************************************************************}

function TBFH.CheckBFHFile(CryptFileName: String): Boolean;
var
  CStream: TFileStream;
begin
  try
    CStream:= TFileStream.Create(CryptFileName, fmOpenRead or fmShareDenyNone);
    try
      Result:= CheckBFHStream(CStream) and (Length(OriginalName) > 0);
    finally
      CStream.Free;
    end;
  except
    Result:= False;
  end;
end;

procedure TBFH.EncryptFile(PlainFileName: String);
var
  PStream: TFileStream;
  CStream: TFileStream;
  CryptFileName: String;
  FPath, FName: String;
  ClearBuffer: Array[0..$3FFF] of Byte;
  i: LongInt;
  FAttr: Integer;
begin
  FAttr:= FileGetAttr(PlainFileName) or faArchive;
  if FileSetAttr(PlainFileName, 0) > 0 then
    raise EFilerError.Create(SysErrorMessage(GetLastError) + ^M^M +
                             PlainFileName);
  if fFastFileCrypt then
  begin
    CryptFileName:= PlainFileName;
    PStream:= TFileStream.Create(PlainFileName, fmOpenRead or fmShareDenyNone);
    CStream:= TFileStream.Create(CryptFileName, fmOpenWrite or fmShareDenyNone);
  end
  else
  begin
    CryptFilename:= ChangeFileExt(PlainFileName, TempExt);
    PStream:= TFileStream.Create(PlainFileName, fmOpenReadWrite or fmShareExclusive);
    CStream:= TFileStream.Create(CryptFileName, fmCreate or fmShareExclusive);
  end;
  try
    OriginalName:= ExtractFileName(PlainFileName);
    OriginalSize:= PStream.Size;
    OriginalDate:= FileGetDate(PStream.Handle);
    EncryptStream(PStream, CStream);
  finally
    PStream.Free;
    CStream.Free;
  end;
  FPath:= ExtractFilePath(CryptFileName);
  if fSecureFileName then
  begin
    i:= 0;
    FName:= GetRandomFileName;
    while not RenameFile(CryptFileName, FPath + FName) do
    begin
      if i < 1000 then
      begin
        FName:= GetRandomFileName;
        Inc(i);
      end
      else
      begin
        if Assigned(fOnRenameFile) then
        begin
          fOnRenameFile(Self, FPath, FName);
          FName:= ExtractFileName(FName);
        end
        else Insert('_', FName, Pos(ExtractFileExt(FName), FName));
      end;
    end;
  end
  else
  begin
    FName:= ExtractFileName(ChangeFileExt(CryptFileName, '.' + BFHSignature));
    while not RenameFile(CryptFileName, FPath + FName) do
    begin
      if Assigned(fOnRenameFile) then
      begin
        fOnRenameFile(Self, FPath, FName);
        FName:= ExtractFileName(FName);
      end
      else Insert('_', FName, Pos(ExtractFileExt(FName), FName));
    end;
  end;
  FileSetAttr(FPath + FName, FAttr);
  if not FastFileCrypt then
  begin
    PStream:= TFileStream.Create(PlainFileName, fmOpenWrite or fmShareExclusive);
    try
      FillChar(ClearBuffer, SizeOf(ClearBuffer), $FF);
      with PStream do
        while Position < Size do
          if (Size - Position) > SizeOf(ClearBuffer) then
            WriteBuffer(ClearBuffer, SizeOf(ClearBuffer))
          else WriteBuffer(ClearBuffer, Size - Position);
      PStream.Position:= 0;
      PStream.Write(BFHSignature, SizeOf(BFHSignature));
      PStream.Size:= PStream.Position;
    finally
      PStream.Free;
    end;
    DeleteFile(PlainFileName);
  end;
end;

function TBFH.DecryptFile(CryptFileName: String): Boolean;
var
  PStream: TFileStream;
  CStream: TFileStream;
  PlainFileName: String;
  FPath, FName: String;
  FAttr: Integer;
begin
  FAttr:= FileGetAttr(CryptFileName) or faArchive;
  if FileSetAttr(CryptFileName, 0) > 0 then
    raise EFilerError.Create(SysErrorMessage(GetLastError) + ^M^M +
                             CryptFileName);
  if FastFileCrypt then
  begin
    PlainFileName:= CryptFileName;
    CStream:= TFileStream.Create(CryptFileName, fmOpenRead or fmShareDenyNone);
    PStream:= TFileStream.Create(PlainFileName, fmOpenWrite or fmShareDenyNone);
  end
  else
  begin
    PlainFileName:= ChangeFileExt(CryptFileName, TempExt);
    CStream:= TFileStream.Create(CryptFileName, fmOpenReadWrite or fmShareExclusive);
    PStream:= TFileStream.Create(PlainFileName, fmCreate or fmShareExclusive);
  end;
  try
    Result:= DecryptStream(CStream, PStream) and (PStream.Size = OriginalSize);
    FileSetDate(PStream.Handle, OriginalDate);
  finally
    CStream.Free;
    PStream.Free;
  end;
  if Result then
  begin
    FPath:= ExtractFilePath(PlainFileName);
    FName:= OriginalName;
    while not RenameFile(PlainFileName, FPath + FName) do
    begin
      if Assigned(fOnRenameFile) then
      begin
        fOnRenameFile(Self, FPath, FName);
        FName:= ExtractFileName(FName);
      end
      else Insert('_', FName, Pos(ExtractFileExt(FName), FName));
    end;
    FileSetAttr(FPath + FName, FAttr);
    if not FastFileCrypt then DeleteFile(CryptFileName);
  end
  else
  begin
    if not FastFileCrypt then DeleteFile(PlainFileName);
  end;
end;

function TBFH.WriteOrigInfo(CryptFileName: String): Boolean;
var
  BFHHeader: TBFHHeader;
  OrigInfo: TOrigInfo;
  CRCBuffer: TCRCBuffer absolute OrigInfo;
  ulCBCLeft, ulCBCRight: LongInt;
  OrgName: ShortString;
  OrgDate: LongInt;
  CStream: TFileStream;
  i: Integer;
begin
  Result:= False;
  OrgName:= OriginalName;
  OrgDate:= OriginalDate;
  if CheckBFHFile(CryptFileName) then
  begin
    OriginalName:= OrgName;
    OriginalDate:= OrgDate;
    CStream:= TFileStream.Create(CryptFileName, fmOpenReadWrite or fmShareDenyNone);
    try
      CStream.Position:= 0;
      CStream.ReadBuffer(BFHHeader, SizeOf(BFHHeader));
      with OrigInfo do
      begin
        for i:= Low(Name) to High(Name) do Name[i]:= Chr(Random($100));
        StrPCopy(Name, OriginalName);
        Size:= OriginalSize;
        Date:= OriginalDate;
      end;
      BFHHeader.CRC32:= GetCRC_32(@CRCBuffer, SizeOf(TCRCBuffer));
      ulCBCLeft:= BFHHeader.ulCBCLeft;
      ulCBCRight:= BFHHeader.ulCBCRight;
      Blowfish_CBCEncrypt(@OrigInfo, SizeOf(TOrigInfo), ulCBCLeft, ulCBCRight);
      CStream.Position:= 0;
      CStream.WriteBuffer(BFHHeader, SizeOf(BFHHeader));
      CStream.WriteBuffer(OrigInfo, SizeOf(OrigInfo));
      Result:= True;
    finally
      CStream.Free;
    end;
  end;
end;

{*** Init *********************************************************************}

procedure Register;
begin
  RegisterComponents('Neu', [TBFH]);
end;

initialization
  Randomize;
end.

